/*
    A customizable name tag to put on a bottle, or Mate-Tag, to prevent confusion which Club Mate bottle belongs to who.
    
    Inspired by everyone who made me jealous at GPN22 and the existing templates on Thingiverse.
    
    Documentation and an tutorial how to create a own logo for the tag can be found of the vspace.one wiki: https://wiki.vspace.one/doku.php?id=projekte:2024:cat-ear-mate-tag
    
    you can contact me via the fediverse: @faye@chaos.social
*/

/* [General] */

// "fixed numbers of fragments in 360 degrees" - basically how many sides circles have to make the model smoother. Value for text and logo is overwritten by 5.
$fn = 100;

// The text that will be on your tag, presumably your name.
name = "Max";

// Font to use for the name. You can use any installed font. For font style use "[font name]:style=[font style]" syntax.
font = "JetBrainsMono";

// Logo on your tag, must be a 50x50mm svg or dxf file. Check the wiki for more information.
logo = "logos/logo_vspaceone_simple.svg";

// Yes they're optional.
ears = true;

/* [Tag] */

// Radius of the bottle neck, is the same for most glass bottles, in mm.
bottle_neck_radius = 13; // [10:0.1:15]

// Changes the shape of the tag to adjust to different bottles, in degrees.
bottle_angle = 12; // [0:0.5:30]

// The height of the tag, may be reduced with no logo to save filament, in mm.
tag_height = 26; // [10:30]

// The thickness of the tag, it shouldn't break but still be flexible, in mm.
tag_thickness = 2; // [1.5:0.1:5]

// The size of the gap to clip your bottle through. in degrees.
gap_angle = 90; // [45:135]

// Text and Logo can either be extruded with a positive value or incised with a negative value, each in mm.
relief = 1; // [-1:0.1:1]

/* [Text/Name] */

// Font size for the name, in mm.
text_size = 8; // [3:20]

// Extra or with negative values less space between the letters, in mm.
text_space = -1; // [-5:0.1:5]

// Moves the text up, or down with negative values, in mm.
text_vertical_move = -6; // [-15:15]

// Moves the text to the right, or left with negative values, in mm.
text_horizontal_move = 0; // [-15:15]

/* [Logo] */

// Size of the chosen logo, in mm.
logo_size = 13; // [5:20]

// Move your logo up, or down with negative values, in mm.
logo_vertical_move = 6; // [-15:15]

// Move your text to the right, or to the left with negative values, in mm.
logo_horizontal_move = 0; // [-15:15]

// Spin the logo around itself, clockwise, in degrees.
logo_spin = 0; // [-180:180]

/* [Ears] */

// Size of the ears, in mm but other values make it appear smaller.
ear_size = 15; // [5:25]

// Thickness of the ears, they don't have to be flexible, in mm.
ear_thickness = 3; // [1:0.1:5]

// Move the ears down into the tag, 0 -> on top of the clipping piece, in mm.
ear_move_down  = 5; // [0:10]

// The position of the ears, relative to the front, in degrees.
ear_angle = 60; // [15:90]

// Rounding the upper edge of the ears, radius, in mm.
ear_rounding = 2.5; // [0:25]

// How far the ears should stretch, orientated at the bottom radius, in mm.
ear_stretch = 0; // [-10:10]


cat_ear_mate_tag(
    name,
    font,
    logo,
    ears,
    bottle_neck_radius,
    bottle_angle,
    tag_height,
    tag_thickness,
    gap_angle,
    relief,
    text_size,
    text_space,
    text_vertical_move,
    text_horizontal_move,
    logo_size,
    logo_vertical_move,
    logo_horizontal_move,
    logo_spin,
    ear_size,
    ear_thickness,
    ear_move_down,
    ear_angle,
    ear_rounding,
    ear_stretch);




module cat_ear_mate_tag(
    name,
    font,
    logo,
    ears,
    bottle_neck_radius,
    bottle_angle,
    tag_height,
    tag_thickness,
    gap_angle,
    relief,
    text_size,
    text_space,
    text_vertical_move,
    text_horizontal_move,
    logo_size,
    logo_vertical_move,
    logo_horizontal_move,
    logo_spin,
    ear_size,
    ear_thickness,
    ear_move_down,
    ear_angle,
    ear_rounding,
    ear_stretch) {
    
    // the actual values for the tag itself
    top_radius = bottle_neck_radius;
    bottom_radius = bottle_neck_radius + (tan(bottle_angle) * tag_height);
    outer_bottom_radius = bottom_radius + tag_thickness;
    outer_top_radius = top_radius + tag_thickness;
    
    // approximates the radius of the tag at the height of the top of the text and logo
    text_radius = outer_top_radius + (outer_bottom_radius - outer_top_radius) * (tag_height - (tag_height / 2 + text_vertical_move + text_size / 2 )) / tag_height;
    
    // just makes sure that the thickness value of the text and logo covers the whole extrusion
    text_thickness = (bottom_radius - top_radius + relief) * 2;
    
    // values for the cylinders which the ears are created with
    ear_top_radius = bottom_radius + ear_stretch + tag_thickness - ear_thickness;
    ear_bottom_radius = top_radius + tag_thickness - ear_thickness;
    ear_outer_top_radius = bottom_radius + ear_stretch + tag_thickness;
    ear_outer_bottom_radius = top_radius + tag_thickness;
    
    // for the empty space in the middle
    void_cone_height = tag_height / ((bottom_radius - top_radius) / bottom_radius);

    // creating the gap to put the tag on the bottle
    difference() {
        // cutting out the hole where the bottle goes
        difference() {
            // adding text and logo relief 
            symmetric_difference() {
                // adding cat ears
                union() {
                    // base cone section
                    cylinder(h = tag_height, r1 = outer_bottom_radius, r2 = outer_top_radius);
                    // making ears optional
                    if(ears == true) {
                        translate([0, 0, tag_height - ear_move_down]) {
                            // intersection between the prisms and the hollow cone section 
                            intersection() {
                                // hollow cone section (area where the ears may be)
                                difference() {
                                    // outer cone section
                                    cylinder(h = ear_size, r1 = ear_outer_bottom_radius, r2 = ear_outer_top_radius);
                                    // inner cone section with extra height and -z so it peeks through both sides
                                    translate([0, 0, -0.01]) cylinder(h = ear_size + 0.02, r1 = ear_bottom_radius, r2 = ear_top_radius);
                                }
                                // 2 prisms that define the shape and size of the ears
                                for ( i = [-1,1]) {
                                    // rotation for each ear in the other direction
                                    rotate([0, -90, 90 + ear_angle * i]) {
                                        // the sections which will be rounded are cut off
                                        intersection() {
                                            // creates the prisms that will eventually become the ears
                                            cylinder(h = ear_outer_top_radius, r = ear_size, $fn=3);
                                            // the cuboid covers the whole ear except what will be rounded
                                            translate([0 , -ear_size, 0]) cube([ear_size - ear_rounding * 1.5, ear_size * 2, ear_outer_top_radius]);
                                        }
                                        // cylinders that are the rounding of the ear
                                        translate([ear_size - ear_rounding * 2, 0, 0]) cylinder(h = ear_outer_top_radius, r = ear_rounding);
                                    }
                                }
                            }
                        }
                    }
                }
                // limiting the text and logo only to the chosen relief
                intersection() {
                    // union between text and logo
                    union() {
                        // text
                        translate([0, 0, (tag_height - text_size) / 2 + text_vertical_move])
                        rotate([90, 0, 0])
                        // for loop to create and move/rotate all letters individually
                        for(letter = [0:1:len(name)]) {
                            // rotate the letter so it falls into it's position with spacing
                            rotate([0, ((letter + 0.5 - len(name)/2) * (text_size + text_space) + text_horizontal_move) / (text_radius * 6.28) * 360, 0])
                            // actually move the letter on the tag
                            translate([0, 0, text_radius])
                            linear_extrude(text_thickness)
                            text(text=name[letter],font=font,size=text_size,halign="center",$fn=5);
                            }
                        // logo
                        translate([0, 0, tag_height / 2 + logo_vertical_move])
                        rotate([90,logo_spin, logo_horizontal_move / (bottom_radius + top_radius * 3.14) * 360])
                        scale([logo_size/50,logo_size/50,1])
                        translate([-25,-25,0])
                        linear_extrude(outer_bottom_radius + relief)
                        import(file=logo,$fn=5);
                    }
                    // hollow cylinder limits the text and logo to the relief
                    symmetric_difference() {
                        cylinder(h = tag_height, r1 = outer_bottom_radius + relief, r2 = outer_top_radius + relief);
                        cylinder(h = tag_height, r1 = outer_bottom_radius, r2 = outer_top_radius);
                    }
                }
            }
        // ensures that the cone peeks through the outer one on the bottom and top
        translate([0,0,-0.01])
        // cone that represents the bottle to create the hole
        cylinder(h = void_cone_height, r1 = bottom_radius, r2 = 0);
        }
    // gap should also go above and beneath the empty cylinder
    translate([0,0,-0.01]) 
    // makes gap polygon 3d
    linear_extrude(tag_height + ear_size + ear_move_down + 0.02)
    // creates the gap polygon with given angle
    polygon([[0,0], [outer_bottom_radius * sin(gap_angle/2), outer_bottom_radius * cos(gap_angle/2)], [outer_bottom_radius, outer_bottom_radius], [0, outer_bottom_radius], [-outer_bottom_radius, outer_bottom_radius], [-outer_bottom_radius * sin(gap_angle/2), outer_bottom_radius * cos(gap_angle/2)]]);
    }
}

module symmetric_difference() {
    union() {
        difference() {
            children(0);
            children(1);
        }
        difference() {
            children(1);
            children(0);
        }
    }
}
        